#include "lpc2138.h"
#include "enc28j60.h"
#include "uip.h"

#define	ENC_CS	0x100000;
#define DISABLE_CS	IO0SET=ENC_CS
#define ENABLE_CS	IO0CLR=ENC_CS


void print(char *fmt, ... );

u8 Enc28j60Bank;
u16 NextPacketPtr;

void nicInit(void)
{
	enc28j60Init();
}

void nicSend(unsigned int len, unsigned char* packet)
{
	enc28j60PacketSend(len, packet);
}

unsigned int nicPoll(unsigned int maxlen, unsigned char* packet)
{
	return enc28j60PacketReceive(maxlen, packet);
}

void nicGetMacAddress(u8* macaddr)
{
	// read MAC address registers
	// NOTE: MAC address in ENC28J60 is byte-backward
	*macaddr++ = enc28j60Read(MAADR5);
	*macaddr++ = enc28j60Read(MAADR4);
	*macaddr++ = enc28j60Read(MAADR3);
	*macaddr++ = enc28j60Read(MAADR2);
	*macaddr++ = enc28j60Read(MAADR1);
	*macaddr++ = enc28j60Read(MAADR0);
}

void nicSetMacAddress(u8* macaddr)
{
	// write MAC address
	// NOTE: MAC address in ENC28J60 is byte-backward
	enc28j60Write(MAADR5, *macaddr++);
	enc28j60Write(MAADR4, *macaddr++);
	enc28j60Write(MAADR3, *macaddr++);
	enc28j60Write(MAADR2, *macaddr++);
	enc28j60Write(MAADR1, *macaddr++);
	enc28j60Write(MAADR0, *macaddr++);
}

void nicRegDump(void)
{
	enc28j60RegDump();
}

u8 enc28j60ReadOp(u8 op, u8 address)
{
	u8 data;
   
	// assert CS
	ENABLE_CS;
	// issue read command
	SSPDR=op | (address & ADDR_MASK);
	// read data
	SSPDR=0x00;
	// do dummy read if needed
	if(address & 0x80)
	{
		SSPDR=0x00;
		//dummy skip
		while(!(SSPSR & 0x04));
		data=SSPDR;
	}
	//duumy skip
	while(!(SSPSR & 0x04));
	data=SSPDR;
	//real data
	while(!(SSPSR & 0x04));
	data=SSPDR;

	// release CS
	DISABLE_CS;

	return data;
}

void enc28j60WriteOp(u8 op, u8 address, u8 data)
{
	// assert CS
	ENABLE_CS;

	// issue write command
	SSPDR=op | (address & ADDR_MASK);
	// write data
	SSPDR=data;
	
	//dummy skip
	while(!(SSPSR & 0x04));
	data=SSPDR;
	while(!(SSPSR & 0x04));
	data=SSPDR;

	// release CS
	DISABLE_CS;
}

void enc28j60ReadBuffer(u16 len, u8* data)
{
	u32 tmp;
	// assert CS
	ENABLE_CS;
	
	// issue read command
	SSPDR=ENC28J60_READ_BUF_MEM;
	//dummy skip
	while(!(SSPSR & 0x04));
	tmp=SSPDR;
	while(len--)
	{
		// read data
		SSPDR=0x00;
		while(!(SSPSR & 0x04));
		*data++=SSPDR;
	}	
	// release CS
	DISABLE_CS;
}

void enc28j60WriteBuffer(u16 len, u8* data)
{
	u32 tmp;
	// assert CS
	ENABLE_CS;
	
	// issue write command
	SSPDR=ENC28J60_WRITE_BUF_MEM;
	//dummy skip
	while(!(SSPSR & 0x04));
	tmp=SSPDR;
	while(len--)
	{
		// write data
		SSPDR=*data++;
		//dummy skip
		while(!(SSPSR & 0x04));
		tmp=SSPDR;
	}	
	// release CS
	DISABLE_CS;
}

void enc28j60SetBank(u8 address)
{
	// set the bank (if needed)
	if((address & BANK_MASK) != Enc28j60Bank)
	{
		// set the bank
		enc28j60WriteOp(ENC28J60_BIT_FIELD_CLR, ECON1, (ECON1_BSEL1|ECON1_BSEL0));
		enc28j60WriteOp(ENC28J60_BIT_FIELD_SET, ECON1, (address & BANK_MASK)>>5);
		Enc28j60Bank = (address & BANK_MASK);
	}
}

u8 enc28j60Read(u8 address)
{
	// set the bank
	enc28j60SetBank(address);
	// do the read
	return enc28j60ReadOp(ENC28J60_READ_CTRL_REG, address);
}

void enc28j60Write(u8 address, u8 data)
{
	// set the bank
	enc28j60SetBank(address);
	// do the write
	enc28j60WriteOp(ENC28J60_WRITE_CTRL_REG, address, data);
}

u16 enc28j60PhyRead(u8 address)
{
	u16 data;

	// Set the right address and start the register read operation
	enc28j60Write(MIREGADR, address);
	enc28j60Write(MICMD, MICMD_MIIRD);

	// wait until the PHY read completes
	while(enc28j60Read(MISTAT) & MISTAT_BUSY);

	// quit reading
	enc28j60Write(MICMD, 0x00);
	
	// get data value
	data  = enc28j60Read(MIRDL);
	data |= enc28j60Read(MIRDH);
	// return the data
	return data;
}

void enc28j60PhyWrite(u8 address, u16 data)
{
	// set the PHY register address
	enc28j60Write(MIREGADR, address);
	
	// write the PHY data
	enc28j60Write(MIWRL, data);	
	enc28j60Write(MIWRH, data>>8);

	// wait until the PHY write completes
	while(enc28j60Read(MISTAT) & MISTAT_BUSY);
}

void enc28j60Init(void)
{
	u32 i;

	PINSEL1=PINSEL1|0x00a8;		//SSP I/O pin enable
	SSPCPSR=0x02;			//SPI CLK = 60MHz/6=10MHz
	SSPCR0=0x0207;			//CPHA=0 CPOL=0
	SSPCR1=0x02;			//MASTER & SSP Enable
	IO0DIR=IO0DIR|ENC_CS;		//CS(P0.20) is output
	DISABLE_CS;			//Disable CS
	
	// perform system reset
	enc28j60WriteOp(ENC28J60_SOFT_RESET, 0, ENC28J60_SOFT_RESET);
	// check CLKRDY bit to see if reset is complete
	//delay_us(50);
	for(i=0;i<100000;){
		i++;
	}
	while(!(enc28j60Read(ESTAT) & ESTAT_CLKRDY));

	// do bank 0 stuff
	// initialize receive buffer
	// 16-bit transfers, must write low byte first
	// set receive buffer start address
	NextPacketPtr = RXSTART_INIT;
	enc28j60Write(ERXSTL, RXSTART_INIT&0xFF);
	enc28j60Write(ERXSTH, RXSTART_INIT>>8);
	// set receive pointer address
	enc28j60Write(ERXRDPTL, RXSTART_INIT&0xFF);
	enc28j60Write(ERXRDPTH, RXSTART_INIT>>8);
	// set receive buffer end
	// ERXND defaults to 0x1FFF (end of ram)
	enc28j60Write(ERXNDL, RXSTOP_INIT&0xFF);
	enc28j60Write(ERXNDH, RXSTOP_INIT>>8);
	// set transmit buffer start
	// ETXST defaults to 0x0000 (beginnging of ram)
	enc28j60Write(ETXSTL, TXSTART_INIT&0xFF);
	enc28j60Write(ETXSTH, TXSTART_INIT>>8);

	// do bank 2 stuff
	// enable MAC receive
	enc28j60Write(MACON1, MACON1_MARXEN|MACON1_TXPAUS|MACON1_RXPAUS);
	// bring MAC out of reset
	enc28j60Write(MACON2, 0x00);
	// enable automatic padding and CRC operations
	enc28j60WriteOp(ENC28J60_BIT_FIELD_SET, MACON3, MACON3_PADCFG0|MACON3_TXCRCEN|MACON3_FRMLNEN);
//	enc28j60Write(MACON3, MACON3_PADCFG0|MACON3_TXCRCEN|MACON3_FRMLNEN);
	// set inter-frame gap (non-back-to-back)
	enc28j60Write(MAIPGL, 0x12);
	enc28j60Write(MAIPGH, 0x0C);
	// set inter-frame gap (back-to-back)
	enc28j60Write(MABBIPG, 0x12);
	// Set the maximum packet size which the controller will accept
	enc28j60Write(MAMXFLL, MAX_FRAMELEN&0xFF);	
	enc28j60Write(MAMXFLH, MAX_FRAMELEN>>8);

	// do bank 3 stuff
	// write MAC address
	// NOTE: MAC address in ENC28J60 is byte-backward
	enc28j60Write(MAADR5, UIP_ETHADDR0);
	enc28j60Write(MAADR4, UIP_ETHADDR1);
	enc28j60Write(MAADR3, UIP_ETHADDR2);
	enc28j60Write(MAADR2, UIP_ETHADDR3);
	enc28j60Write(MAADR1, UIP_ETHADDR4);
	enc28j60Write(MAADR0, UIP_ETHADDR5);

	// no loopback of transmitted frames
	enc28j60PhyWrite(PHCON2, PHCON2_HDLDIS);

	// switch to bank 0
	enc28j60SetBank(ECON1);
	// enable interrutps
	enc28j60WriteOp(ENC28J60_BIT_FIELD_SET, EIE, EIE_INTIE|EIE_PKTIE);
	// enable packet reception
	enc28j60WriteOp(ENC28J60_BIT_FIELD_SET, ECON1, ECON1_RXEN);
/*
	enc28j60PhyWrite(PHLCON, 0x0AA2);

	// setup duplex ----------------------

	// Disable receive logic and abort any packets currently being transmitted
	enc28j60WriteOp(ENC28J60_BIT_FIELD_CLR, ECON1, ECON1_TXRTS|ECON1_RXEN);
	
	{
		u16 temp;
		// Set the PHY to the proper duplex mode
		temp = enc28j60PhyRead(PHCON1);
		temp &= ~PHCON1_PDPXMD;
		enc28j60PhyWrite(PHCON1, temp);
		// Set the MAC to the proper duplex mode
		temp = enc28j60Read(MACON3);
		temp &= ~MACON3_FULDPX;
		enc28j60Write(MACON3, temp);
	}

	// Set the back-to-back inter-packet gap time to IEEE specified 
	// requirements.  The meaning of the MABBIPG value changes with the duplex
	// state, so it must be updated in this function.
	// In full duplex, 0x15 represents 9.6us; 0x12 is 9.6us in half duplex
	//enc28j60Write(MABBIPG, DuplexState ? 0x15 : 0x12);	
	
	// Reenable receive logic
	enc28j60WriteOp(ENC28J60_BIT_FIELD_SET, ECON1, ECON1_RXEN);

	// setup duplex ----------------------
*/
}

void enc28j60PacketSend(unsigned int len, unsigned char* packet)
{
	//Errata: Transmit Logic reset
	enc28j60WriteOp(ENC28J60_BIT_FIELD_SET, ECON1, ECON1_TXRST);
	enc28j60WriteOp(ENC28J60_BIT_FIELD_CLR, ECON1, ECON1_TXRST);

	// Set the write pointer to start of transmit buffer area
	enc28j60Write(EWRPTL, TXSTART_INIT);
	enc28j60Write(EWRPTH, TXSTART_INIT>>8);
	// Set the TXND pointer to correspond to the packet size given
	enc28j60Write(ETXNDL, (TXSTART_INIT+len));
	enc28j60Write(ETXNDH, (TXSTART_INIT+len)>>8);

	// write per-packet control byte
	enc28j60WriteOp(ENC28J60_WRITE_BUF_MEM, 0, 0x00);

	// copy the packet into the transmit buffer
	enc28j60WriteBuffer(len, packet);
	
	// send the contents of the transmit buffer onto the network
	enc28j60WriteOp(ENC28J60_BIT_FIELD_SET, ECON1, ECON1_TXRTS);
}

void enc28j60PacketSend2(unsigned int len1, unsigned char* packet1, unsigned int len2, unsigned char* packet2)
{
	//Errata: Transmit Logic reset
	enc28j60WriteOp(ENC28J60_BIT_FIELD_SET, ECON1, ECON1_TXRST);
	enc28j60WriteOp(ENC28J60_BIT_FIELD_CLR, ECON1, ECON1_TXRST);

	// Set the write pointer to start of transmit buffer area
	enc28j60Write(EWRPTL, TXSTART_INIT);
	enc28j60Write(EWRPTH, TXSTART_INIT>>8);
	// Set the TXND pointer to correspond to the packet size given
	enc28j60Write(ETXNDL, (TXSTART_INIT+len1+len2));
	enc28j60Write(ETXNDH, (TXSTART_INIT+len1+len2)>>8);

	// write per-packet control byte
	enc28j60WriteOp(ENC28J60_WRITE_BUF_MEM, 0, 0x00);

	// copy the packet into the transmit buffer
	enc28j60WriteBuffer(len1, packet1);
	if(len2>0) enc28j60WriteBuffer(len2, packet2);
	
	// send the contents of the transmit buffer onto the network
	enc28j60WriteOp(ENC28J60_BIT_FIELD_SET, ECON1, ECON1_TXRTS);
}

unsigned int enc28j60PacketReceive(unsigned int maxlen, unsigned char* packet)
{
	u16 rxstat;
	u16 len;

	// check if a packet has been received and buffered
	if( !(enc28j60Read(EIR) & EIR_PKTIF) )
		return 0;
	
	// Make absolutely certain that any previous packet was discarded	
	//if( WasDiscarded == FALSE)
	//	MACDiscardRx();

	// Set the read pointer to the start of the received packet
	enc28j60Write(ERDPTL, (NextPacketPtr));
	enc28j60Write(ERDPTH, (NextPacketPtr)>>8);
	// read the next packet pointer
	NextPacketPtr  = enc28j60ReadOp(ENC28J60_READ_BUF_MEM, 0);
	NextPacketPtr |= enc28j60ReadOp(ENC28J60_READ_BUF_MEM, 0)<<8;
	// read the packet length
	len  = enc28j60ReadOp(ENC28J60_READ_BUF_MEM, 0);
	len |= enc28j60ReadOp(ENC28J60_READ_BUF_MEM, 0)<<8;
	// read the receive status
	rxstat  = enc28j60ReadOp(ENC28J60_READ_BUF_MEM, 0);
	rxstat |= enc28j60ReadOp(ENC28J60_READ_BUF_MEM, 0)<<8;

	// limit retrieve length
	// (we reduce the MAC-reported length by 4 to remove the CRC)
	if(len>maxlen){
		len=maxlen;
	}

	// copy the packet from the receive buffer
	enc28j60ReadBuffer(len, packet);

	// Move the RX read pointer to the start of the next received packet
	// This frees the memory we just read out
	enc28j60Write(ERXRDPTL, (NextPacketPtr));
	enc28j60Write(ERXRDPTH, (NextPacketPtr)>>8);

	// decrement the packet counter indicate we are done with this packet
	enc28j60WriteOp(ENC28J60_BIT_FIELD_SET, ECON2, ECON2_PKTDEC);

	return len;
}

void enc28j60ReceiveOverflowRecover(void)
{
	// receive buffer overflow handling procedure

	// recovery completed
}

void enc28j60RegDump(void)
{
	print("RevID: 0x%x\r\n", enc28j60Read(EREVID));

	print("Cntrl:\n");
	print("ECON1=%x ",enc28j60Read(ECON1));
	print("ECON2=%x ",enc28j60Read(ECON2));
	print("ESTAT=%x ",enc28j60Read(ESTAT));
	print("EIR=%x\n",enc28j60Read(EIR));
	print("EIE=%x\n",enc28j60Read(EIE));

	print("MAC:\n");
	print("MACON1=%x ",enc28j60Read(MACON1));
	print("MACON2=%x ",enc28j60Read(MACON2));
	print("MACON3=%x ",enc28j60Read(MACON3));
	print("MACON4=%x\n",enc28j60Read(MACON4));
	print("MAD=%x%x\n",
		enc28j60Read(MAADR5)*0x1000000+enc28j60Read(MAADR4)*0x10000+enc28j60Read(MAADR3)*0x100+enc28j60Read(MAADR2),
		enc28j60Read(MAADR1)*0x1000000+enc28j60Read(MAADR0)*0x10000+0xffff);

	print("Rx:\n");
	print("ERXST=%x ",enc28j60Read(ERXSTH)*0x100+enc28j60Read(ERXSTL));
	print("ERXND=%x ",enc28j60Read(ERXNDH)*0x100+enc28j60Read(ERXNDL));
	print("ERXWRPT=%x ",enc28j60Read(ERXWRPTH)*0x100+enc28j60Read(ERXWRPTL));
	print("ERXRDPT=%x\n",enc28j60Read(ERXRDPTH)*0x100+enc28j60Read(ERXRDPTL));
	print("ERXFCON=%x ",enc28j60Read(ERXFCON));
	print("EPKTCNT=%x ",enc28j60Read(EPKTCNT));
	print("MAMXFL=%x ",enc28j60Read(MAMXFLH)*0x100+enc28j60Read(MAMXFLL));

	print("Tx:\n");
	print("ETXST=%x ",enc28j60Read(ETXSTH)*0x100+enc28j60Read(ETXSTL));
	print("ETXND=%x ",enc28j60Read(ETXNDH)*0x100+enc28j60Read(ETXNDL));
	print("MACLCON1=%x ",enc28j60Read(MACLCON1));
	print("MACLCON2=%x\n",enc28j60Read(MACLCON2));
	print("MAPHSUP=%x\n",enc28j60Read(MAPHSUP));
}



